/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/***

Copyright (c) 1997  Hewlett-Packard

Module Name:
    Pci32Reg.c	(PCI Win32 Device Registry Access functions)

Abstract:
    This file includes all functions necessary to access the Win32 registry
    during operations (open, close etc.) using the b_ntpci device driver.

Notes:
    ** IMPORTANT **
    This module is Unicode-compliant.
    Use _TCHAR instead of char, _T and _tcs... macros to keep it that way.

Environment:
    Win32 ONLY (!!!)

Revision History:
    Original; SCRobinson, 5/97

***/


#include "IdentOs.h"
#include "Pci32Reg.h"

/* lengths include a safety factor to accomodate different code versions */

/* max length of the version string */
#define MAX_LENGTH_OF_VER       (sizeof(CAPI_VERSION) + \
                                 sizeof(_TCHAR) + (10 * sizeof(_TCHAR)))
/* max length of the registry path */
#define MAX_LENGTH_OF_PATH      (sizeof(CAPI_HPBEST_DEVICES_PATH) + \
                                 sizeof(REG_BASE_DEVICE_NAME) + (10 * sizeof(_TCHAR)))

/* For initializing strings; do NOT use '\0' !!!! */
#define  ZERO_TERM_STR   ((_TCHAR)0)

/* private functions */
static b_errtype BestNTRegKeyOpen( b_int32 * pdwDeviceEnum, PHKEY phKey );

static void BestNTRegConcat( 
					        _TCHAR * pcszResult,			/* caller's buffer */
					        const _TCHAR * pcszString1,
					        const _TCHAR * pcszString2,
					        const _TCHAR * pcszString3,
					        b_int32 * dwNumber
					        );

/*
/////////////////////////////////////////////////////////////////////// 
// Returns B_E_OK if C-API and Driver versions agree AND we're running
// under Windows NT (as long as we don't have drivers for other OS's), 
// else error code.
// ** ALWAYS returns B_E_OK in NT debug mode ** 
*/
b_errtype BestNTRegVersionCheck( void )
{
#ifdef NO_VERSION_CHECK // _DEBUG
    /* SCR; 13/6/97; added check for Win95 */
    if( OS_WINNT != GetOsVersionShort() )
        return B_E_PCI_NT_ONLY;

    return B_E_OK;
#else

    static _TCHAR  cDriverVerBuff[MAX_LENGTH_OF_VER] = {ZERO_TERM_STR};
    static BOOLEAN fIsWinNT = FALSE;
    static BOOLEAN fIsVerOK = FALSE;

    b_int32 dwBufferSize = MAX_LENGTH_OF_VER;
    b_errtype b_Ret;

    /* the fast way */
    if( fIsVerOK )
        return B_E_OK;

    /* SCR; 13/6/97; added check for Win95 */
    if( !fIsWinNT )
    {
        fIsWinNT = (BOOLEAN)(( OS_WINNT == GetOsVersionShort() ) || ( OS_WIN95 == GetOsVersionShort() ));
        if( !fIsWinNT )
            return B_E_PCI_NT_ONLY;
    }

    /* get and compare driver version (in registry) with C-API version */
    /* do this only if we failed the last time */
    if( !fIsVerOK || !*cDriverVerBuff )
    {
        if( B_E_OK != (b_Ret = BestNTRegSzGet( VALUENAME_VERSION, 
                                              DEV_ENUM_COMMON,
                                              cDriverVerBuff,
                                              &dwBufferSize))
          ) return b_Ret;

        if( 0 != _tcscmp(cDriverVerBuff, CAPI_VERSION) )
            return B_E_DRIVER_VERSION_DIFF;

        fIsVerOK = TRUE;
    }

    return B_E_OK;

#endif  /* #ifdef  _DEBUG */
}


/*
/////////////////////////////////////////////////////////////////////// 
// Enumerates through Registered PCI devices (see b_ntpci) and returns
// the HPSlotID of the matching device 
*/
b_errtype BestNTRegDevIDGet( 
							b_int32 dwVendorId,
							b_int32 dwDeviceId,
							b_int32 dwSubsysId,
							b_int32 *pdwHPSlotId
							)
{
    b_int32     dwNumDevices;
    b_int32     dwDevNum;
    b_int32     dwData;
    b_int32     dwEnumFoundEntry = 0;
    b_errtype   b_Ret;

    /* just in case... */
    if( !pdwHPSlotId )
        return B_E_REG_NULL_PTR_PARAM;

    /* init for failure */
    *pdwHPSlotId = INVALID_HPSLOTID;

    /* get the number of devices that we're dealing with */
    if( B_E_OK != (b_Ret = BestNTRegDWordGet( VALUENAME_DEV_COUNT, 
                                             DEV_ENUM_COMMON,
                                             &dwNumDevices))
      ) return b_Ret;

    /* loop through each entry */
    for( dwDevNum = 0; dwDevNum < dwNumDevices; dwDevNum++ )
    {
        /* check vendor id */
        if( B_E_OK != (b_Ret = BestNTRegDWordGet( VALUENAME_VENDORID, 
                                                 dwDevNum,
                                                 &dwData))
          ) return b_Ret;

        if( dwData != dwVendorId )
            continue;
             
        /* vendor id good...check device id */
        if( B_E_OK != (b_Ret = BestNTRegDWordGet( VALUENAME_DEVICEID, 
                                                 dwDevNum,
                                                 &dwData))
          ) return b_Ret;

        if( dwData != dwDeviceId )
            continue;

        /* found it...is this THE one? */
        if( dwEnumFoundEntry == dwSubsysId )
        {
            return BestNTRegDWordGet( VALUENAME_HPSLOTID, 
                                      dwDevNum,
                                      pdwHPSlotId);
        }

        /* on to the next one */
        ++dwEnumFoundEntry; 
    }

	return B_E_NO_BEST_PCI_DEVICE_FOUND;
}


/*
/////////////////////////////////////////////////////////////////////// 
// Returns the "CreateFile()" device driver name from the registry for 
// a particular HPSlotId 
*/
b_errtype BestNTRegDevNameGet( 
                            b_int32 dwHPSlotId,
                            _TCHAR * pszUserBuffer,
                            b_int32 * pdwBufferSize     /* returned val = # of bytes read */
							)
{
    b_int32     dwDevNum;
    b_errtype   b_Ret;

    /* just in case... */
    if( !pszUserBuffer || !pdwBufferSize )
        return B_E_REG_NULL_PTR_PARAM;

    /* init for failure */
    *pszUserBuffer = ZERO_TERM_STR;

    /* get the index of the device that we're looking for */
    if( B_E_OK != (b_Ret = BestNTRegDevEnumGet( dwHPSlotId, &dwDevNum )))
	    return b_Ret;

    /* slot id good...pass back name (or error code) */
    return BestNTRegSzGet( VALUENAME_DEVICENAME, 
                           dwDevNum,
                           pszUserBuffer,
                           pdwBufferSize);
}


/*
/////////////////////////////////////////////////////////////////////// 
// Returns the device driver index in the registry for passed HPSlotId 
*/
b_errtype BestNTRegDevEnumGet( 
                            b_int32 dwHPSlotId,
                            b_int32 * pdwDevNum     /* returned; index of dev. */
							)
{

#define MAX_DEVS    50

    static b_int32 dwNumDevices = 0;
    static b_int32 adwHPSlotIds[MAX_DEVS];
    b_int32     dwDevNum;
    b_int32     dwData;
    b_errtype   b_Ret;

    /* just in case... */
    if( !pdwDevNum )
        return B_E_REG_NULL_PTR_PARAM;

    /* init for failure */
    *pdwDevNum = DEV_ENUM_NOT_FOUND;

    /* one-time initialization of the array */
    if( 0 == dwNumDevices )
    {
        /* init ALL elements of the array */
        for( dwDevNum = 0; dwDevNum < MAX_DEVS; dwDevNum++ )
        {
            adwHPSlotIds[dwDevNum] = INVALID_HPSLOTID;
        }

        /* get the number of drivers */
        if( B_E_OK != (b_Ret = BestNTRegDWordGet( VALUENAME_DEV_COUNT, 
                                                 DEV_ENUM_COMMON,
                                                 &dwNumDevices))
          ) return b_Ret;

        /* loop through each entry */
        for( dwDevNum = 0; dwDevNum < dwNumDevices; dwDevNum++ )
        {
            /* get HPSlotId */
            if( B_E_OK != (b_Ret = BestNTRegDWordGet( VALUENAME_HPSLOTID, 
                                                     dwDevNum,
                                                     &dwData))
              ) return b_Ret;

            /* store in array for fast retrieval later on */
            adwHPSlotIds[dwDevNum] = dwData;
        }
    }

    /* loop through each array entry */
    for( dwDevNum = 0; dwDevNum < dwNumDevices; dwDevNum++ )
    {
        if( adwHPSlotIds[dwDevNum] == dwHPSlotId )
        {
            /* slot id found...pass back index */
            *pdwDevNum = dwDevNum;
            return B_E_OK;
        }
    }

	return B_E_NO_BEST_PCI_DEVICE_FOUND;
}


/*
/////////////////////////////////////////////////////////////////////// 
// Claims the passed device number as belonging to the current process. 
// Returns B_E_OK on success or other error code from BestNTRegOwnerClear() 
*/
b_errtype BestNTRegOwnerSet(
                            b_int32 dwHPSlotId,
                            b_int32 * pdwCurrOwnerProcessId
							)
{
    b_int32     dwDevNum;
    b_errtype   b_Ret;

    /* just in case... */
    if( !pdwCurrOwnerProcessId )
        return B_E_REG_NULL_PTR_PARAM;

    /* clear the current owner id first...2 good returns from this call */
    /* B_E_REG_OWNER_NONE or B_E_OK (it belonged to us anyway) */
    b_Ret = BestNTRegOwnerClear( dwHPSlotId, pdwCurrOwnerProcessId );

    if( (b_Ret != B_E_OK) && (b_Ret != B_E_REG_OWNER_NONE) )
        return b_Ret;
        
    /* it's clear...claim it! */
    /* get the index of the device that we're looking for */
    if( B_E_OK != (b_Ret = BestNTRegDevEnumGet( dwHPSlotId, &dwDevNum )))
	    return b_Ret;

    /* set the current process as the owner but dont' change */
    /* the caller's variable unless we're 100% sure that the */
    /* change has been registered. */
    if( B_E_OK != (b_Ret = BestNTRegDWordSet( VALUENAME_OWNERID,
                                             dwDevNum,
                                             (b_int32) GetCurrentProcessId() ))
      ) return b_Ret;

    *pdwCurrOwnerProcessId = (b_int32) GetCurrentProcessId();

    return B_E_OK;
}


/*
/////////////////////////////////////////////////////////////////////// 
// Releases the passed device (no longer belongs to current process)
// Returns; B_E_OK on success.
//			B_E_REG_OWNER_IS_OTHER if the current process is not the owner 
//			B_E_REG_OWNER_NONE if the device had no owner.
*/
b_errtype BestNTRegOwnerClear(
                            b_int32 dwHPSlotId,
                            b_int32 * pdwCurrOwnerProcessId
							)
{
    b_int32     dwDevNum;
    b_errtype   b_Ret;

    /* just in case... */
    if( !pdwCurrOwnerProcessId )
        return B_E_REG_NULL_PTR_PARAM;

    /* init for failure */
    *pdwCurrOwnerProcessId = DEVICE_OWNER_UNKNOWN;

    /* get the index of the device that we're looking for */
    if( B_E_OK != (b_Ret = BestNTRegDevEnumGet( dwHPSlotId, &dwDevNum )))
	    return b_Ret;

    /* Get the existing owner */
    if( B_E_OK != (b_Ret = BestNTRegDWordGet( VALUENAME_OWNERID,
                                             dwDevNum,
                                             pdwCurrOwnerProcessId ))
      ) return b_Ret;

    /* return if there is no owner (but this warrants an error code) */
    if( DEVICE_HAS_NO_OWNER == *pdwCurrOwnerProcessId )
        return B_E_REG_OWNER_NONE;

    /* device has an owner...is it this process? */
    if( *pdwCurrOwnerProcessId != (b_int32) GetCurrentProcessId() )
        return B_E_REG_OWNER_IS_OTHER;

    /* it belonged to us...clear it! */
    if( B_E_OK != (b_Ret = BestNTRegDWordSet( VALUENAME_OWNERID,
                                             dwDevNum,
                                             DEVICE_HAS_NO_OWNER ))
      ) return b_Ret;

    *pdwCurrOwnerProcessId = DEVICE_HAS_NO_OWNER;

    return B_E_OK;
}


/*	
///////////////////////////////////////////////////////////////////////
// Returns the contents of any DWORD value (use defines in b_ntpci.h)
*/
b_errtype BestNTRegDWordGet( 
							const _TCHAR * pcszValueName,
							b_int32 dwDeviceEnum,
							b_int32 *pdwValue
							)
{
	HKEY        hKey;
    LONG        lRet;
    b_errtype   b_Ret;

	ULONG   ulData;
	ULONG   ulSize = sizeof( ULONG );
	ULONG   ulType;


    /* just in case... */
    if( !pcszValueName || !pdwValue )
        return B_E_REG_NULL_PTR_PARAM;

    if( B_E_OK != (b_Ret = BestNTRegKeyOpen( &dwDeviceEnum, &hKey )))
	    return b_Ret;

    lRet = RegQueryValueEx(
			                hKey,					/* handle of key to query  */
			                pcszValueName,			/* address of name of value to query  */
			                NULL,					/* reserved  */
			                &ulType,				/* address of buffer for value type  */
			                (LPBYTE)&ulData,		/* address of data buffer  */
			                &ulSize 				/* address of data buffer size  */
			                );
    
    DBG_ApiSuccessIsZero(lRet);

	(void)RegCloseKey( hKey );

    if( ERROR_SUCCESS != lRet )             /* Couldn't get value */
	    return B_E_REG_VALUE_NOT_GET;

    /* Check that returned Value really is a REG_DWORD */
	if( (REG_DWORD != ulType) || (sizeof( ULONG ) != ulSize) )		
		return B_E_REG_VALUE_NOT_DWORD;

	*pdwValue = ulData;
	return B_E_OK;
}


/*
///////////////////////////////////////////////////////////////////////
// Sets any DWORD value (use defines in b_ntpci.h)
*/
b_errtype BestNTRegDWordSet( 
							const _TCHAR * pcszValueName,
							b_int32 dwDeviceEnum,
							b_int32 dwValue
							)
{
	HKEY        hKey;
    b_errtype   b_Ret;
    LONG        lRet;

    /* just in case... */
    if( !pcszValueName )
        return B_E_REG_NULL_PTR_PARAM;

    if( B_E_OK != (b_Ret = BestNTRegKeyOpen( &dwDeviceEnum, &hKey )))
	    return b_Ret;

	lRet = RegSetValueEx(
			            hKey,					/* handle of key to query  */
			            pcszValueName,			/* address of name of value to query  */
			            0,					    /* reserved  */
			            REG_DWORD,				/* address of buffer for value type  */
			            (CONST BYTE *)&dwValue,	/* address of data buffer  */
			            sizeof( DWORD ) 		/* address of data buffer size  */
			            );

    DBG_ApiSuccessIsZero(lRet);

	(void)RegCloseKey( hKey );

    if( ERROR_SUCCESS != lRet )     /* Couldn't set value */
	    return B_E_REG_VALUE_NOT_SET;

	return B_E_OK;
}


/*
///////////////////////////////////////////////////////////////////////
// Returns the contents of any STRING value (use defines in b_ntpci.h)
*/
b_errtype BestNTRegSzGet( 
						const _TCHAR * pcszValueName,
						b_int32 dwDeviceEnum,
						_TCHAR * pszUserBuffer,
                        b_int32 * pdwBufferSize     /* returned val = # of bytes read */
						)
{
	HKEY        hKey;
    LONG        lRet;
    b_errtype   b_Ret;

	ULONG   ulType;

    /* just in case... */
    if( !pcszValueName || !pszUserBuffer || !pdwBufferSize )
        return B_E_REG_NULL_PTR_PARAM;

    /* init the buffer for the worst case. */
	*pszUserBuffer = ZERO_TERM_STR;

    /* open the right key */
    if( B_E_OK != (b_Ret = BestNTRegKeyOpen( &dwDeviceEnum, &hKey )))
	    return b_Ret;

	lRet = RegQueryValueEx(
			                hKey,					/* handle of key to query  */
			                pcszValueName,			/* address of name of value to query  */
			                NULL,					/* reserved  */
			                &ulType,				/* address of buffer for value type  */
			                (LPBYTE)pszUserBuffer,  /* address of data buffer  */
			                pdwBufferSize 			/* address of data buffer size  */
			                );

    DBG_ApiSuccessIsZero(lRet);

	(void)RegCloseKey( hKey );

    if( ERROR_SUCCESS != lRet )         /* Couldn't get value */
	    return B_E_REG_VALUE_NOT_GET;

	if( REG_SZ != ulType )				/* data type is not REG_SZ */
		return B_E_REG_VALUE_NOT_SZ;

	return B_E_OK;

}

/* 
///////////////////////////////////////////////////////////////////////
// Single place to open any of the PCI device subkeys.
// You MUST close the key if the return value is B_E_OK
*/ 
static b_errtype BestNTRegKeyOpen(
                                  b_int32 * pdwDeviceEnum,
                                  PHKEY phKey
                                 )
{
    /* buffer for registry subkey names */
    static _TCHAR cBuff[MAX_LENGTH_OF_PATH];

    /* just in case... */
    if( !pdwDeviceEnum || !phKey )
        return B_E_REG_NULL_PTR_PARAM;

    if( *pdwDeviceEnum == DEV_ENUM_COMMON )
        pdwDeviceEnum = NULL;

    /* init for failure.. */
    *phKey = NULL;

	/* make the subkey name */
	BestNTRegConcat( cBuff,
					 CAPI_HPBEST_DEVICES_PATH, 
					 BACK_SLASH, 
					 REG_BASE_DEVICE_NAME, 
					 pdwDeviceEnum );

    if( ERROR_SUCCESS != RegOpenKeyEx(
		                              HKEY_LOCAL_MACHINE,	/* handle of open key  */
		                              cBuff,			    /* address of name of subkey to open  */
		                              0,					/* reserved  */
		                              KEY_ALL_ACCESS,		/* security access mask  */
		                              phKey 				/* address of handle of open key  */
	                                  ) )
    {
        DBG_ApiLastError;
        return B_E_REG_KEY_NOT_OPEN;
    }

    return B_E_OK;
}


/*
///////////////////////////////////////////////////////////////////////
// Concatenates passed strings and a number.  Used to get a device
// subkey with a number at the end.
*/
static void BestNTRegConcat( 
					        _TCHAR * pcszResult,			/* caller's buffer */
					        const _TCHAR * pcszString1,
					        const _TCHAR * pcszString2,
					        const _TCHAR * pcszString3,
					        b_int32 * dwNumber
					        )
{
	_TCHAR NumberBuffer[11] = {ZERO_TERM_STR};

    /* everything EXCEPT this one can be NULL */
    if( !pcszResult )
        return;

	*pcszResult = ZERO_TERM_STR;

	if( pcszString1 )
		_tcscat( pcszResult, pcszString1 );

	if( pcszString2 )
		_tcscat( pcszResult, pcszString2 );

	if( pcszString3 )
		_tcscat( pcszResult, pcszString3 );

	if( dwNumber )
	{
		_stprintf( NumberBuffer, "%u", *dwNumber );
		_tcscat( pcszResult, NumberBuffer );
	}

	return;
}

